/*****************************************************************************
*
*	PerspectiveEngine.cpp : the implementation file for the CPerspectiveEngine class
*
*	Original programming:	Winchell Chung	10/20/1999
*	Changes:				
*
*=============== Copyright (C) 1999 Winchell D. Chung, Jr. ===============*/

#include "stdafx.h"
#include "PerspectiveEngine.h"
#include <math.h>

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

/*****************************************************************************
*	Operations
*****************************************************************************/

//////////////////////////////////////////////////////////////////////////////
//
// This class converts three dimensional points in virtual space
// into 2 dimensional co-ordinates in perspective on the drawing screen.
// The virtual location of the "eyes" viewing the 3 dimensional points
// is controlled with setTheta(), setPhi() and setRho().
// initEye() sets which eye (left or right) is the currently selected eye.
// setEyeSeparation() controls how widely spaced the eyes are.
// setViewAngle() controls how much of an angle the drawing screen 
// encompasses (fish-eye or normal eye).
//
// Once the above functions have set the parameters, one
// makes calls to transformPoint() for every three dimensional point
// in the scene.
//
// The protected functions have the mathematical guts of this
// routine, and contain lots of difficult equations.
//
// **** WARNING!!! ****
// if you are compiling this from MSVC++ 4.2 or earlier, do NOT use the
// full optimization switch ( /Ox ) without using the -Op switch.
// Otherwise the trigonmetric functions will return incorrect values.

/*****************************************************************************
*	Constructors
*****************************************************************************/
CPerspectiveEngine::CPerspectiveEngine(double initialTheta, double initialPhi, double initialRho)
{
 	double	delta_x, delta_y, delta_z;

	from[LEFTEYE].x = -0.25;
	from[LEFTEYE].y = 0.0;
	from[LEFTEYE].z = 6;
	from[RIGHTEYE].x = 0.25;
	from[RIGHTEYE].y = 0.0;
	from[RIGHTEYE].z = 6;
	at.x = 0.0;
	at.y = 0.0;
	at.z = 0.0;
	up.x = 0.0;
	up.y = 1.0;
	up.z = 0.0;

	center.x = (from[LEFTEYE].x + from[RIGHTEYE].x) / 2 ;
	center.y = (from[LEFTEYE].y + from[RIGHTEYE].y) / 2 ;
	center.z = (from[LEFTEYE].z + from[RIGHTEYE].z) / 2 ;

	delta_x = from[LEFTEYE].x - from[RIGHTEYE].x;
	delta_y = from[LEFTEYE].y - from[RIGHTEYE].y;
	delta_z = from[LEFTEYE].z - from[RIGHTEYE].z;
	separation = sqrt((delta_x * delta_x) + (delta_y * delta_y) + (delta_z * delta_z));

	thetaRadians = initialTheta * DEGREES_TO_RADIANS;
	thetaDegrees = initialTheta;
	rhoValue = initialRho;
	phiRadians = initialPhi * DEGREES_TO_RADIANS;
	phiDegrees= initialPhi;

	origin.x = 0.0;
	origin.y = 0.0;
	origin.z = 0.0;

    setViewAngle(45.0);

	polarToRectangular();

	initEye(LEFTEYE);
}

/*****************************************************************************
*	Destructors
*****************************************************************************/
CPerspectiveEngine::~CPerspectiveEngine()
{
}

/*****************************************************************************
*	Operations
*****************************************************************************/

/***************************************************************************/
// initEye
// Sets the which eye is the current eye, LEFTEYE or RIGHTEYE
// All subsequent calls to transformPoint() will use this eye for the calculation.
/***************************************************************************/
void CPerspectiveEngine::initEye(int eyeID)
{
	aPrime = subtract(&at, &from[eyeID]);
	aPrimeMag = mag(&aPrime);

	a3 = divide(&aPrime, aPrimeMag);

	temp = cross(&aPrime, &up);
	tempMag = mag(&temp);

	a1 = divide(&temp, tempMag);

#ifdef OLD
	temp = Cross(&a1, &a3);
	tempMag = Mag(&temp);
#else
	temp = cross(&aPrime, &up);
	temp = cross(&temp, &aPrime);
	tempMag = mag(&temp);
#endif

	a2 = divide(&temp, tempMag);

	offX = -a1.x * from[eyeID].x - a1.y * from[eyeID].y - a1.z * from[eyeID].z;
	offY = -a2.x * from[eyeID].x - a2.y * from[eyeID].y - a2.z * from[eyeID].z;
	offZ = -a3.x * from[eyeID].x - a3.y * from[eyeID].y - a3.z * from[eyeID].z;
}

/***************************************************************************/
// transformPoint
// Transforms the virtual 3-dimensional point into the drawing screen's
// two dimensional point, showing the point in perspective from the current eye.
/***************************************************************************/
void CPerspectiveEngine::transformPoint(TriPoint* theLoc, double* ex, double* ey)
{
	TriPoint	eyeC;
	TriPoint	originShiftLoc;

	originShiftLoc.x = theLoc->x - origin.x;
	originShiftLoc.y = theLoc->y - origin.y;
	originShiftLoc.z = theLoc->z - origin.z;

	eyeC.x =
		(originShiftLoc.x * a1.x) +
		(originShiftLoc.y * a1.y) +
		(originShiftLoc.z * a1.z) +
		offX;
	eyeC.y =
		(originShiftLoc.x * a2.x) +
		(originShiftLoc.y * a2.y) +
		(originShiftLoc.z * a2.z) +
		offY;
	eyeC.z =
		(originShiftLoc.x * a3.x) +
		(originShiftLoc.y * a3.y) +
		(originShiftLoc.z * a3.z) +
		offZ;

	*ex = eyeC.x * dval;
	if (eyeC.z != 0.0)	
		*ex = *ex / eyeC.z;

	*ey = eyeC.y * dval;
	if (eyeC.z != 0.0)	
		*ey = *ey / eyeC.z;
}

/***************************************************************************/
void CPerspectiveEngine::setTheta(double newThetaDegrees)
{
	thetaDegrees = normalizeDegrees(newThetaDegrees);
	thetaRadians = thetaDegrees * DEGREES_TO_RADIANS;
	polarToRectangular();
}

/***************************************************************************/
void CPerspectiveEngine::incrTheta()
{
	setTheta(ratchetDegrees(thetaDegrees, 1.0));
}

/***************************************************************************/
void CPerspectiveEngine::decrTheta()
{
    setTheta(ratchetDegrees(thetaDegrees, -1.0));
}

/***************************************************************************/
void CPerspectiveEngine::deltaTheta(BOOL isIncr)
{
    if (isIncr) incrTheta();
    else        decrTheta();
}

/***************************************************************************/
void CPerspectiveEngine::setPhi(double newPhiDegrees)
{
	phiDegrees = normalizeDegrees(newPhiDegrees);
	phiRadians = phiDegrees * DEGREES_TO_RADIANS;
	polarToRectangular();
}

/***************************************************************************/
void CPerspectiveEngine::incrPhi()
{
    setPhi(ratchetDegrees(phiDegrees, 1.0));
}

/***************************************************************************/
void CPerspectiveEngine::decrPhi()
{
    setPhi(ratchetDegrees(phiDegrees, -1.0));
}

/***************************************************************************/
void CPerspectiveEngine::deltaPhi(BOOL isIncr)
{
    if (isIncr) incrPhi();
    else        decrPhi();
}

/***************************************************************************/
void CPerspectiveEngine::setRho(double newRho)
{
	rhoValue = newRho;
	polarToRectangular();
}

/***************************************************************************/
void CPerspectiveEngine::incrRho()
{
    rhoValue++;
}

/***************************************************************************/
void CPerspectiveEngine::decrRho()
{
	rhoValue--;
	if (rhoValue < 0.0)
		rhoValue = 0.0;
}

/***************************************************************************/
void CPerspectiveEngine::deltaRho(BOOL isIncr)
{
	if (isIncr) incrRho();
	else        decrRho();
}

/***************************************************************************/
void CPerspectiveEngine::setViewAngle(double newAngle)
{
	viewAngleDegrees = normalizeDegrees(newAngle);
	viewAngleRadians = viewAngleDegrees * DEGREES_TO_RADIANS;
	dval = cos(viewAngleRadians / 2) / sin(viewAngleRadians / 2);
}

/***************************************************************************/
void CPerspectiveEngine::setEyeSeparation(double newSeparation)
{
	separation = newSeparation;
	polarToRectangular();
}

/***************************************************************************/


/*****************************************************************************
*	Implementation
*****************************************************************************/
/***************************************************************************/
double CPerspectiveEngine::normalizeDegrees(double theDegrees)
{
	double	normalDegrees = theDegrees;

	while (normalDegrees < 0.0) 
		normalDegrees += 360.0;

	while (normalDegrees > 360.0) 
		normalDegrees -= 360.0;

	return normalDegrees;
}

/***************************************************************************/
double CPerspectiveEngine::ratchetDegrees(double initialDegrees, double ratchetValue)
{
    return normalizeDegrees(initialDegrees + ratchetValue);
}

/***************************************************************************/
// Method mag
// Vector Magnitude operation
/***************************************************************************/
double CPerspectiveEngine::mag(TriPoint *v)
{
#ifdef OLD
	return fabs((v->x * v->x) + (v->y * v->y) + (v->z * v->z));
#else
	return sqrt((v->x * v->x) + (v->y * v->y) + (v->z * v->z));
#endif
}

/***************************************************************************/
// Method subtract
// Vector Subtraction
/***************************************************************************/
TriPoint CPerspectiveEngine::subtract(TriPoint *v1, TriPoint *v2)
{
	TriPoint	c;

	c.x = v1->x - v2->x;
	c.y = v1->y - v2->y;
	c.z = v1->z - v2->z;

	return c;
}

/***************************************************************************/
// Method cross
// Cross Product
/***************************************************************************/
TriPoint CPerspectiveEngine::cross(TriPoint *v1, TriPoint *v2)
{
	TriPoint	c;

	c.x = v1->y * v2->z - v2->y * v1->z;
	c.y = v1->z * v2->x - v2->z * v1->x;
	c.z = v1->x * v2->y - v2->x * v1->y;

	return c;
}

/***************************************************************************/
// Method divide
// Vector division
/***************************************************************************/
TriPoint CPerspectiveEngine::divide(TriPoint *v, double num)
{
	TriPoint	result;

	result.x = v->x;
	result.y = v->y;
	result.z = v->z;

	if (num != 0)
	{
		result.x = v->x / num;
		result.y = v->y / num;
		result.z = v->z / num;
	}

	return result;
}

/***************************************************************************/
// Method polarToRectangular
/***************************************************************************/
void CPerspectiveEngine::polarToRectangular()
{
	int		quadrant;
	double	hyp;

	center.x = rhoValue * sin(phiRadians) * cos(thetaRadians);
	center.y = rhoValue * sin(phiRadians) * sin(thetaRadians);
	center.z = rhoValue * cos(phiRadians);

	if (center.x >= 0)
	{
		if (center.y >= 0)	quadrant = 1;	// QII  | QI
		else				quadrant = 4;	// -,+  | +,+
	}										// -----+-----
	else									// QIII | QIV 
	{										// -,-  | +,- 
		if (center.y >= 0)	quadrant = 2;
		else				quadrant = 3;
	}

	// convert the "between-the-eyes" point and the separation into left & right eye co-ords
	switch (quadrant)
	{
	case 1:
		from[LEFTEYE].x  = center.x + (sin(thetaRadians) * separation);
		from[LEFTEYE].y  = center.y - (cos(thetaRadians) * separation);
		from[RIGHTEYE].x = center.x - (sin(thetaRadians) * separation);
		from[RIGHTEYE].y = center.y + (cos(thetaRadians) * separation);
		break;
	case 2:
		from[LEFTEYE].x  = center.x + (sin(ONE_EIGHTY - thetaRadians) * separation);
		from[LEFTEYE].y  = center.y + (cos(ONE_EIGHTY - thetaRadians) * separation);
		from[RIGHTEYE].x = center.x - (sin(ONE_EIGHTY - thetaRadians) * separation);
		from[RIGHTEYE].y = center.y - (cos(ONE_EIGHTY - thetaRadians) * separation);
		break;
	case 3:
		from[LEFTEYE].x  = center.x - (sin(thetaRadians - ONE_EIGHTY) * separation);
		from[LEFTEYE].y  = center.y + (cos(thetaRadians - ONE_EIGHTY) * separation);
		from[RIGHTEYE].x = center.x + (sin(thetaRadians - ONE_EIGHTY) * separation);
		from[RIGHTEYE].y = center.y - (cos(thetaRadians - ONE_EIGHTY) * separation);
		break;
	case 4:
		from[LEFTEYE].x  = center.x - (sin(THREE_SIXTY - thetaRadians) * separation);
		from[LEFTEYE].y  = center.y - (cos(THREE_SIXTY - thetaRadians) * separation);
		from[RIGHTEYE].x = center.x + (sin(THREE_SIXTY - thetaRadians) * separation);
		from[RIGHTEYE].y = center.y + (cos(THREE_SIXTY - thetaRadians) * separation);
		break;
	}
	from[LEFTEYE].z = center.z;
	from[RIGHTEYE].z = center.z;

	// Look through the centerpoint at a point about 100x away
	at.x = ( -center.x ) * 100.0;
	at.y = ( -center.y ) * 100.0;
	at.z = ( -center.z ) * 100.0;

	// Do something arcane to approximate the "up" value
	up.z = sin(phiRadians) * 10.0;
	hyp  = cos(phiRadians) * 10.0;
	up.x = -(cos(thetaRadians) * hyp);
	up.y = -(sin(thetaRadians) * hyp);
}

/***************************************************************************/
